<?php 
/*
Plugin Name: WooCommerce old products image remover 
Description: This plugin allows you to delete images associated with product sheets upon deletion. Whether it is the image in front, the photo galleries or the images linked to the variations.
Author: Wycan
Version: 1.0
WC requires at least: 4.0.0
WC tested up to: 9.3.3
Text Domain: wyc-wopir
Author URI: https://wycan.fr
*/

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Main class for the WooCommerce Old Products Image Remover plugin
 */
class WYC_WooCommerce_Old_Products_Image_Remover {

    /**
     * Constructor to initialize the plugin
     */
    public function __construct() {
        // Check if WooCommerce is installed and active
        add_action('admin_init', [$this, 'wopir_check_woocommerce_installed']);
        
        // Ensure compatibility with WooCommerce HPOS
        add_action('before_woocommerce_init', [$this, 'wopir_check_hpos_compatibility']);
        
        // Load plugin textdomain for translations
        add_action('plugins_loaded', [$this, 'wopir_load_plugin_textdomain']);
        
        // Add admin menu
        add_action('admin_menu', [$this, 'wycan_add_admin_page']);
    }

    /**
     * Check if WooCommerce is installed and active
     */
    public function wopir_check_woocommerce_installed() {
        include_once(ABSPATH . 'wp-admin/includes/plugin.php');
        if (!is_plugin_active('woocommerce/woocommerce.php')) {
            add_action('admin_notices', [$this, 'wopir_woocommerce_missing_notice']);
        } else {
            add_action("before_delete_post", [$this, 'wopir_delete_product_attached_files'], 10, 1);
        }
    }

    /**
     * Display an admin notice if WooCommerce is not installed
     */
    public function wopir_woocommerce_missing_notice() {
        ?>
        <div class="notice notice-error">
            <p><?php esc_html_e('WooCommerce is required to use the WooCommerce old products image remover plugin.', 'wyc-wopir'); ?></p>
            <p>
                <a href="<?php echo esc_url(admin_url('plugin-install.php?s=woocommerce&tab=search&type=term')); ?>" class="button button-primary">
                    <?php esc_html_e('Download WooCommerce', 'wyc-wopir'); ?>
                </a>
            </p>
        </div>
        <?php
    }

    /**
     * Delete images attached to a WooCommerce product when it is deleted
     */
    public function wopir_delete_product_attached_files($post_id) {
        if (get_post_type($post_id) !== 'product') {
            return;
        }
        
        // Get all attachments linked to the product
        $args = array(
            'post_parent' => $post_id,
            'post_type'   => 'attachment', 
            'numberposts' => -1,
            'post_status' => 'any' 
        );
        
        $attachments = get_children($args);
        
        if (!empty($attachments)) {
            foreach ($attachments as $attachment) {  
                wp_delete_attachment($attachment->ID, true);
            }
        }
        
        global $wpdb;
        
        // Delete variation images
        $variation_image_ids = $wpdb->get_col($wpdb->prepare("
            SELECT pm.meta_value 
            FROM {$wpdb->postmeta} pm 
            JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
            WHERE p.post_parent = %d 
            AND p.post_type = 'product_variation' 
            AND pm.meta_key = '_thumbnail_id' 
            AND pm.meta_value != ''
        ", $post_id));
        
        if (!empty($variation_image_ids)) {
            foreach ($variation_image_ids as $image_id) {
                if (!empty($image_id) && is_numeric($image_id)) {
                    wp_delete_attachment($image_id, true);
                }
            }
        }
        
        // Clear meta values for variation images
        $wpdb->query($wpdb->prepare("
            UPDATE {$wpdb->postmeta} pm 
            JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
            SET pm.meta_value = '' 
            WHERE p.post_parent = %d 
            AND p.post_type = 'product_variation' 
            AND pm.meta_key = '_thumbnail_id'
        ", $post_id));
        
        // Clear additional variation images
        $wpdb->query($wpdb->prepare("
            UPDATE {$wpdb->postmeta} pm 
            JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
            SET pm.meta_value = '' 
            WHERE p.post_parent = %d 
            AND p.post_type = 'product_variation' 
            AND pm.meta_key = '_wc_additional_variation_images'
        ", $post_id));
    }

    /**
     * Ensure compatibility with WooCommerce HPOS (High-Performance Order Storage)
     */
    public function wopir_check_hpos_compatibility() {
        if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        }
    }

    /**
     * Load the plugin textdomain for translations
     */
    public function wopir_load_plugin_textdomain() {
        load_plugin_textdomain('wyc-wopir', false, dirname(plugin_basename(__FILE__)) . '/languages/');
    }

    /**
     * Add a submenu page under the Settings menu
     */
    public function wycan_add_admin_page() {
        // Add a submenu page if not already added
        if (!function_exists('wycan_add_admin_page')) {
            add_submenu_page(
                'options-general.php',
                'Wycan Plugins',
                'Wycan',
                'manage_options',
                'wycan-admin',
                [$this, 'wycan_admin_page_callback']
            );
        }
    }

    /**
     * Callback function to display the admin page content
     */
    public function wycan_admin_page_callback() {
        // Display the admin page content if not already defined
        if (!function_exists('wycan_admin_page_callback')) {
            ?>
            <iframe style="height:100vh;width:100%;border:none" 
                    src="https://wycan.fr/admin-plugins-wordpress/" 
                    width="100%" height="100vh" 
                    title="Wycan plugin"></iframe>
            <?php 
        }
    }
}

// Initialize the plugin
new WYC_WooCommerce_Old_Products_Image_Remover();

/**
* Add custom action links to the plugin in the plugins list
*/
function wopir_admin_action_links($links) {
    $links[] = '<a href="' . admin_url('admin.php?page=wycan-admin') . '">' . esc_html__('Our plugins', 'wyc-wopir') . '</a>';
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'wopir_admin_action_links');